/**
 * Frontend JavaScript for the Custom Drill-Down Filter.
 *
 * @version 8.0.4
 * - CRITICAL FIX: Rewrites filter application logic to use the correct JetSmartFilters API, resolving the console error.
 * - FEATURE: Restores drill-down functionality by parsing the full category tree from the `data-category-tree` attribute.
 * - REMOVED: AJAX calls for sub-categories are no longer needed, improving performance.
 */
(function($) {
    'use strict';

    const CustomDrillDownFilter = function($scope) {
        const $widget = $scope.find('.custom-drilldown-filter-widget');
        if (!$widget.length || $widget.data('initialized')) {
            return;
        }
        $widget.data('initialized', true);

        // --- DOM Elements ---
        const $container = $widget.find('.custom-drilldown-container');
        const $backButton = $container.find('.drilldown-back-button');
        const $removeAllButton = $container.find('.drilldown-remove-all');
        const $title = $container.find('.drilldown-current-title');
        const $levelsWrapper = $container.find('.drilldown-levels-wrapper');

        // --- Filter Settings ---
        const providerID = $widget.data('smart-filters-provider');
        const filterID = $widget.data('filter-id');
        const queryVar = $widget.data('query-var');
        const queryID = $widget.data('query-id') || 'default';
        const applyType = $widget.data('apply-type') || 'ajax';
        const isAjax = 'ajax' === applyType;
        const initialParentId = $widget.data('initial-parent-id');
        const initialTitle = $title.text();

        // --- State Management ---
        let historyState = []; // Stores { termId, termName } for navigation
        let categoryTree = {};
        const rawTreeData = $widget.attr('data-category-tree'); // Use .attr() to get the raw string
        
        try {
            categoryTree = rawTreeData ? JSON.parse(rawTreeData) : {};
        } catch (e) {
            console.error('Drilldown Filter: Failed to parse category data.', e, rawTreeData);
            $levelsWrapper.html('<p>Error: Could not load category data.</p>');
            return;
        }
        
        /**
         * FIX: Applies the filter using the correct JetSmartFilters API.
         */
        const applyFilter = (termId) => {
            const filterGroup = window.JetSmartFilters.filterGroups[providerID + '/' + queryID];

            if (!filterGroup) {
                console.warn('JetSmartFilters: Filter group not found for ' + providerID + '/' + queryID);
                return;
            }
            
            // For taxonomy filters, the filter ID is the query var.
            const targetFilter = filterGroup.getFilter(queryVar);

            if (targetFilter) {
                targetFilter.setValue(termId);
                filterGroup.applyFilters();
            } else {
                 console.warn(`JetSmartFilters: Filter with query_var "${queryVar}" not found in group.`);
            }
        };

        /**
         * FIX: Resets the filter using the correct API.
         */
        const resetFilter = () => {
            const filterGroup = window.JetSmartFilters.filterGroups[providerID + '/' + queryID];
            if (!filterGroup) return;

            const targetFilter = filterGroup.getFilter(queryVar);
            if (targetFilter) {
                targetFilter.resetValue();
                filterGroup.applyFilters();
            }
        };

        /**
         * Renders a list of categories for a given parent object from the tree.
         */
        const renderLevel = (levelData, levelIndex) => {
            const $newLevel = $(`<div class="drilldown-level" data-level="${levelIndex}"></div>`);
            if ($.isEmptyObject(levelData)) {
                 $newLevel.append('<p>No sub-categories found.</p>');
            } else {
                $.each(levelData, (id, term) => {
                    const hasChildren = term.children && !$.isEmptyObject(term.children);
                    const itemClass = hasChildren ? 'drilldown-item has-children' : 'drilldown-item';
                    $newLevel.append(`<div class="${itemClass}" data-term-id="${term.term_id}" data-term-name="${term.name}"><span>${term.name}</span></div>`);
                });
            }
            
            $levelsWrapper.children().last().hide();
            $levelsWrapper.append($newLevel);
        };
        
        const drillDown = (termId, termName) => {
            let children = null;
            // Find the term in the tree to get its children
            function findChildren(tree, id) {
                if (tree[id]) return tree[id].children;
                for (const key in tree) {
                    if (tree[key].children) {
                        const found = findChildren(tree[key].children, id);
                        if (found) return found;
                    }
                }
                return null;
            }
            children = findChildren(categoryTree, termId);

            if (children) {
                historyState.push({ termId, termName });
                renderLevel(children, historyState.length);
                $title.text(termName);
                updateButtonsVisibility();
            }
        };

        const goBack = () => {
            if (historyState.length === 0) return;
            
            historyState.pop();
            $levelsWrapper.children().last().remove();
            $levelsWrapper.children().last().show();
            
            const previousState = historyState.length > 0 ? historyState[historyState.length - 1] : null;
            
            if (previousState) {
                $title.text(previousState.termName);
            } else {
                $title.text(initialTitle);
            }
            updateButtonsVisibility();
        };

        const updateButtonsVisibility = () => {
            const filterGroup = window.JetSmartFilters.filterGroups[providerID + '/' + queryID];
            let hasActiveFilters = false;

            if (filterGroup) {
                const targetFilter = filterGroup.getFilter(queryVar);
                if (targetFilter) {
                    hasActiveFilters = !!targetFilter.getValue();
                }
            }
            
            $removeAllButton.toggle(hasActiveFilters);
            $backButton.toggle(historyState.length > 0);
        };

        // --- Event Listeners ---
        $levelsWrapper.on('click', '.drilldown-item', function() {
            const $this = $(this);
            const termId = $this.data('term-id');
            const termName = $this.data('term-name');
            applyFilter(termId);
            drillDown(termId, termName);
        });

        $backButton.on('click', function(e) {
            e.preventDefault();
            goBack();
        });

        $removeAllButton.on('click', function(e) {
            e.preventDefault();
            resetFilter();
            historyState = [];
            $levelsWrapper.children().not(':first').remove();
            $levelsWrapper.children(':first').show();
            $title.text(initialTitle);
            updateButtonsVisibility();
        });

        // Initial render
        renderLevel(categoryTree, 0);

        // Sync button visibility with JetSmartFilters state changes.
        window.JetSmartFilters.events.subscribe('jet-smart-filters/inited', updateButtonsVisibility);
        window.JetSmartFilters.events.subscribe('ajaxFilters/updated', updateButtonsVisibility);
    };

    $(window).on('elementor/frontend/init', function() {
        elementorFrontend.hooks.addAction('frontend/element_ready/custom_drilldown_filter.default', CustomDrillDownFilter);
    });

})(jQuery);
