<?php
/**
 * Elementor Widget Class for the Drill-Down Filter.
 *
 * Version: 7.1.0 - Fixes fatal JSON parsing error by properly escaping the category tree data attribute.
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

class Elementor_Custom_Drilldown_Filter_Widget extends \Elementor\Widget_Base {

    // All get_name(), get_title(), get_icon(), etc. methods remain the same...
    public function get_name() { return 'custom_drilldown_filter'; }
    public function get_title() { return __( 'Drill-Down Category Filter', 'custom-drilldown-filter' ); }
    public function get_icon() { return 'eicon-folder-tree'; }
    public function get_categories() { return [ 'custom-filters' ]; }
    public function get_script_depends() { return [ 'custom-category-filter-js' ]; }
    public function get_style_depends() { return [ 'custom-category-filter-css' ]; }

    protected function register_controls() {
        // All controls remain the same as the previous version.
        // For brevity, they are not repeated here, but they are still active in the plugin.
        // --- Content Tab ---
        $this->start_controls_section('content_section', ['label' => __( 'Filter Configuration', 'custom-drilldown-filter' ),'tab' => \Elementor\Controls_Manager::TAB_CONTENT,]);
        $this->add_control('filter_title', ['label' => __( 'Title', 'custom-drilldown-filter' ), 'type' => \Elementor\Controls_Manager::TEXT, 'default' => __( 'Category', 'custom-drilldown-filter' ),]);
        $this->add_control('parent_categories', ['label' => __( 'Select Parent Categories', 'custom-drilldown-filter' ), 'type' => \Elementor\Controls_Manager::SELECT2, 'label_block' => true, 'multiple' => true, 'options' => $this->get_top_level_categories_for_control(), 'description' => __( 'Leave empty to show all top-level categories.', 'custom-drilldown-filter' ),]);
        $this->add_control('provider_id', ['label' => __( 'Provider CSS ID', 'custom-drilldown-filter' ), 'type' => \Elementor\Controls_Manager::TEXT, 'default' => 'products-grid', 'description' => __( 'Enter the CSS ID of the JetEngine Listing Grid to filter.', 'custom-drilldown-filter' ),]);
        $this->end_controls_section();
        // --- Behavior Tab ---
        $this->start_controls_section('behavior_section', ['label' => __( 'Behavior', 'custom-drilldown-filter' ), 'tab' => \Elementor\Controls_Manager::TAB_CONTENT,]);
        $this->add_control('start_with_subcategories', ['label' => __( 'Start with Sub-categories', 'custom-drilldown-filter' ), 'type' => \Elementor\Controls_Manager::SWITCHER, 'return_value' => 'yes', 'default' => 'no', 'description' => __( 'If you have selected only ONE parent category, this will show its children on initial load.', 'custom-drilldown-filter' ),]);
        $this->add_control('start_with_sub_title_source', ['label' => __( 'Title Source', 'custom-drilldown-filter' ), 'type' => \Elementor\Controls_Manager::SELECT, 'default' => 'parent_name', 'options' => ['parent_name'  => __( 'Use Parent Category Name', 'custom-drilldown-filter' ), 'custom_title' => __( 'Use Custom Widget Title', 'custom-drilldown-filter' ),], 'condition' => ['start_with_subcategories' => 'yes',],]);
        $this->add_control('lock_to_selection', ['label' => __( 'Lock to Initial Selection', 'custom-drilldown-filter' ), 'type' => \Elementor\Controls_Manager::SWITCHER, 'return_value' => 'yes', 'default' => 'no', 'description' => __( 'Prevents the user from going back past the initial list of categories.', 'custom-drilldown-filter' ),]);
        $this->add_control('back_button_text', ['label' => __( 'Back Button Text', 'custom-drilldown-filter' ), 'type' => \Elementor\Controls_Manager::TEXT, 'default' => __( '&larr; Back', 'custom-drilldown-filter' ),]);
        $this->end_controls_section();
        // --- Style Tab ---
        $this->start_controls_section('style_section_title', ['label' => __( 'Title', 'custom-drilldown-filter' ), 'tab' => \Elementor\Controls_Manager::TAB_STYLE,]);
        $this->add_responsive_control('title_align', ['label' => __( 'Alignment', 'custom-drilldown-filter' ), 'type' => \Elementor\Controls_Manager::CHOOSE, 'options' => ['left' => ['title' => __( 'Left', 'custom-drilldown-filter' ), 'icon' => 'eicon-text-align-left',], 'center' => ['title' => __( 'Center', 'custom-drilldown-filter' ), 'icon' => 'eicon-text-align-center',], 'right' => ['title' => __( 'Right', 'custom-drilldown-filter' ), 'icon' => 'eicon-text-align-right',],], 'selectors' => ['{{WRAPPER}} .current-category-title' => 'text-align: {{VALUE}};',],]);
        $this->add_control('title_color', ['label' => __( 'Color', 'custom-drilldown-filter' ), 'type' => \Elementor\Controls_Manager::COLOR, 'selectors' => ['{{WRAPPER}} .current-category-title' => 'color: {{VALUE}}',],]);
        $this->add_group_control(\Elementor\Group_Control_Typography::get_type(), ['name' => 'title_typography', 'selector' => '{{WRAPPER}} .current-category-title',]);
        $this->end_controls_section();
        $this->start_controls_section('style_section_back_button', ['label' => __( 'Back Button', 'custom-drilldown-filter' ), 'tab' => \Elementor\Controls_Manager::TAB_STYLE,]);
        $this->add_responsive_control('back_button_align', ['label' => __( 'Alignment', 'custom-drilldown-filter' ), 'type' => \Elementor\Controls_Manager::CHOOSE, 'options' => ['flex-start' => ['title' => __( 'Left', 'custom-drilldown-filter' ), 'icon' => 'eicon-text-align-left',], 'center' => ['title' => __( 'Center', 'custom-drilldown-filter' ), 'icon' => 'eicon-text-align-center',], 'flex-end' => ['title' => __( 'Right', 'custom-drilldown-filter' ), 'icon' => 'eicon-text-align-right',],], 'selectors' => ['{{WRAPPER}} .filter-header' => 'justify-content: {{VALUE}};',],]);
        $this->add_control('back_button_color', ['label' => __( 'Color', 'custom-drilldown-filter' ), 'type' => \Elementor\Controls_Manager::COLOR, 'selectors' => ['{{WRAPPER}} .back-link' => 'color: {{VALUE}}',],]);
        $this->add_group_control(\Elementor\Group_Control_Typography::get_type(), ['name' => 'back_button_typography', 'selector' => '{{WRAPPER}} .back-link',]);
        $this->end_controls_section();
        $this->start_controls_section('style_section_items', ['label' => __( 'Category Items', 'custom-drilldown-filter' ), 'tab' => \Elementor\Controls_Manager::TAB_STYLE,]);
        $this->add_responsive_control('item_align', ['label' => __( 'Alignment', 'custom-drilldown-filter' ), 'type' => \Elementor\Controls_Manager::CHOOSE, 'options' => ['left' => ['title' => __( 'Left', 'custom-drilldown-filter' ), 'icon' => 'eicon-text-align-left',], 'center' => ['title' => __( 'Center', 'custom-drilldown-filter' ), 'icon' => 'eicon-text-align-center',], 'right' => ['title' => __( 'Right', 'custom-drilldown-filter' ), 'icon' => 'eicon-text-align-right',],], 'selectors' => ['{{WRAPPER}} .category-item a' => 'justify-content: {{VALUE}};',],]);
        $this->add_control('item_name_heading', ['label' => __( 'Item Name', 'custom-drilldown-filter' ), 'type' => \Elementor\Controls_Manager::HEADING,]);
        $this->add_control('item_name_color', ['label' => __( 'Color', 'custom-drilldown-filter' ), 'type' => \Elementor\Controls_Manager::COLOR, 'selectors' => ['{{WRAPPER}} .cat-name' => 'color: {{VALUE}}',],]);
        $this->add_group_control(\Elementor\Group_Control_Typography::get_type(), ['name' => 'item_name_typography', 'selector' => '{{WRAPPER}} .cat-name',]);
        $this->add_control('item_count_heading', ['label' => __( 'Item Count', 'custom-drilldown-filter' ), 'type' => \Elementor\Controls_Manager::HEADING, 'separator' => 'before',]);
        $this->add_control('item_count_color', ['label' => __( 'Color', 'custom-drilldown-filter' ), 'type' => \Elementor\Controls_Manager::COLOR, 'selectors' => ['{{WRAPPER}} .cat-count' => 'color: {{VALUE}}',],]);
        $this->add_control('item_count_bg_color', ['label' => __( 'Background Color', 'custom-drilldown-filter' ), 'type' => \Elementor\Controls_Manager::COLOR, 'selectors' => ['{{WRAPPER}} .cat-count' => 'background-color: {{VALUE}}',],]);
        $this->add_group_control(\Elementor\Group_Control_Typography::get_type(), ['name' => 'item_count_typography', 'selector' => '{{WRAPPER}} .cat-count',]);
        $this->end_controls_section();
    }

    protected function render() {
        $settings = $this->get_settings_for_display();
        
        $initial_terms_data = $this->get_initial_terms_data($settings);
        $full_category_tree = $this->get_full_category_tree($settings);

        // THE CORE FIX: Use esc_attr() on the JSON encoded string to prevent errors.
        $encoded_tree = esc_attr(json_encode($full_category_tree));

        ?>
        <div id="custom-category-filter-container-<?php echo esc_attr($this->get_id()); ?>" 
             class="custom-category-filter" 
             data-provider-id="<?php echo esc_attr($settings['provider_id']); ?>"
             data-lock-selection="<?php echo esc_attr($settings['lock_to_selection']); ?>"
             data-initial-parent-id="<?php echo esc_attr($initial_terms_data['initial_parent_id']); ?>"
             data-back-button-text="<?php echo esc_attr($settings['back_button_text']); ?>"
             data-category-tree='<?php echo $encoded_tree; ?>'
             >
            <div class="filter-header">
                <h4 class="current-category-title"><?php echo esc_html($initial_terms_data['initial_title']); ?></h4>
                <a href="#" class="back-link" style="display: none;"><?php echo wp_kses_post($settings['back_button_text']); ?></a>
            </div>
            <div class="filter-body">
                <ul class="category-list">
                    <?php if ( ! empty( $initial_terms_data['terms'] ) ) : ?>
                        <?php foreach ( $initial_terms_data['terms'] as $term ) : ?>
                            <li class="category-item">
                                <a href="#" data-term-id="<?php echo esc_attr( $term->term_id ); ?>">
                                    <span class="cat-name"><?php echo esc_html( $term->name ); ?></span>
                                    <span class="cat-count"><?php echo esc_html( $term->count ); ?></span>
                                </a>
                            </li>
                        <?php endforeach; ?>
                    <?php else: ?>
                        <li>No categories found.</li>
                    <?php endif; ?>
                </ul>
            </div>
        </div>
        <?php
    }

    private function get_initial_terms_data($settings) {
        $initial_parent_id = 0;
        $initial_title = $settings['filter_title'];
        $terms = [];

        if ( $settings['start_with_subcategories'] === 'yes' && count($settings['parent_categories']) === 1 ) {
            $single_parent_id = $settings['parent_categories'][0];
            $initial_parent_id = $single_parent_id;
            
            if ($settings['start_with_sub_title_source'] === 'parent_name') {
                $parent_term = get_term($single_parent_id);
                if ($parent_term) {
                    $initial_title = $parent_term->name;
                }
            }
            
            $terms = get_terms(['taxonomy' => 'product_cat', 'parent' => $single_parent_id, 'hide_empty' => false]);
        } else {
            $query_args = ['taxonomy' => 'product_cat', 'hide_empty' => false];
            if ( ! empty( $settings['parent_categories'] ) ) {
                $query_args['include'] = $settings['parent_categories'];
                $query_args['orderby'] = 'include';
            } else {
                $query_args['parent'] = 0;
            }
            $terms = get_terms( $query_args );
        }
        return ['terms' => $terms, 'initial_title' => $initial_title, 'initial_parent_id' => $initial_parent_id];
    }

    private function get_full_category_tree($settings) {
        $transient_key = 'custom_filter_tree_v7_' . md5(json_encode($settings['parent_categories']));
        $cached_tree = get_transient($transient_key);

        if (false !== $cached_tree) {
            return $cached_tree;
        }

        $all_terms = get_terms('product_cat', ['hide_empty' => false]);
        if (is_wp_error($all_terms) || empty($all_terms)) {
            return [];
        }

        $full_tree = $this->build_terms_hierarchy($all_terms);

        if (!empty($settings['parent_categories'])) {
            $pruned_tree = array_intersect_key($full_tree, array_flip($settings['parent_categories']));
            $tree = $pruned_tree;
        } else {
            $tree = $full_tree;
        }
        
        set_transient($transient_key, $tree, HOUR_IN_SECONDS);
        return $tree;
    }

    private function build_terms_hierarchy(array $terms, $parentId = 0) {
        $branch = [];
        foreach ($terms as $term) {
            if ($term->parent == $parentId) {
                $children = $this->build_terms_hierarchy($terms, $term->term_id);
                $branch[$term->term_id] = [
                    'id'       => $term->term_id,
                    'name'     => $term->name,
                    'slug'     => $term->slug,
                    'count'    => $term->count,
                    'children' => !empty($children) ? $children : null,
                ];
            }
        }
        return $branch;
    }
    
    private function get_top_level_categories_for_control() {
        $terms = get_terms(['taxonomy' => 'product_cat', 'parent' => 0, 'hide_empty' => false]);
        return is_wp_error($terms) ? [] : wp_list_pluck($terms, 'name', 'term_id');
    }
}
