<?php
/**
 * Settings Page for the Drill-Down Filter.
 *
 * This file creates a settings page under the "Tools" menu in the WordPress admin area.
 * It uses the WordPress Settings API to create, register, and save options.
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

class Custom_Filter_Settings_Page {

    public function __construct() {
        add_action( 'admin_menu', [ $this, 'add_admin_menu' ] );
        add_action( 'admin_init', [ $this, 'register_settings' ] );
    }

    /**
     * Add the menu item to the WordPress admin menu.
     */
    public function add_admin_menu() {
        add_submenu_page(
            'tools.php',                                  // Parent slug
            __( 'Drill-Down Filter Settings', 'custom-drilldown-filter' ), // Page title
            __( 'Drill-Down Filter', 'custom-drilldown-filter' ),      // Menu title
            'manage_options',                             // Capability
            'custom_drilldown_filter_settings',           // Menu slug
            [ $this, 'render_settings_page' ]              // Callback function
        );
    }

    /**
     * Register the settings, sections, and fields.
     */
    public function register_settings() {
        // Register a setting group
        register_setting(
            'custom_filter_options_group', // Option group
            'custom_filter_hide_empty'     // Option name
        );

        // Add a settings section
        add_settings_section(
            'custom_filter_main_section', // ID
            __( 'Global Filter Settings', 'custom-drilldown-filter' ), // Title
            null, // Callback
            'custom_drilldown_filter_settings' // Page
        );

        // Add a settings field
        add_settings_field(
            'hide_empty_categories', // ID
            __( 'Hide Empty Categories', 'custom-drilldown-filter' ), // Title
            [ $this, 'render_hide_empty_field' ], // Callback
            'custom_drilldown_filter_settings', // Page
            'custom_filter_main_section' // Section
        );
    }

    /**
     * Render the checkbox field for the "Hide Empty" setting.
     */
    public function render_hide_empty_field() {
        $option = get_option( 'custom_filter_hide_empty' );
        ?>
        <input type="checkbox" id="custom_filter_hide_empty" name="custom_filter_hide_empty" value="1" <?php checked( 1, $option, true ); ?> />
        <label for="custom_filter_hide_empty"><?php _e( 'Check this to hide categories that have no products.', 'custom-drilldown-filter' ); ?></label>
        <?php
    }

    /**
     * Render the HTML for the settings page.
     */
    public function render_settings_page() {
        ?>
        <div class="wrap">
            <h1><?php echo esc_html( get_admin_page_title() ); ?></h1>
            <form action="options.php" method="post">
                <?php
                // Output security fields for the registered setting group
                settings_fields( 'custom_filter_options_group' );
                // Output the settings sections and their fields
                do_settings_sections( 'custom_drilldown_filter_settings' );
                // Output the save button
                submit_button( __( 'Save Settings', 'custom-drilldown-filter' ) );
                ?>
            </form>
        </div>
        <?php
    }
}

new Custom_Filter_Settings_Page();
