<?php
/**
 * Plugin Name:       Custom WooCommerce Checkout
 * Description:       3‑step checkout UI via REST + native Woo payment. Admin settings, region limits, Field Builder driven fields.
 * Version:           2.5.0
 * Author:            Your Name
 * License:           GPL-2.0+
 * Text Domain:       custom-checkout
 */

if ( ! defined( 'ABSPATH' ) ) exit;

if ( ! defined( 'CC_PLUGIN_FILE' ) ) define( 'CC_PLUGIN_FILE', __FILE__ );
if ( ! defined( 'CC_PLUGIN_DIR' ) )  define( 'CC_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
if ( ! defined( 'CC_PLUGIN_URL' ) )  define( 'CC_PLUGIN_URL', plugin_dir_url( __FILE__ ) );

function cc_safe_include( $rel_path, $require = true ) {
    $path = CC_PLUGIN_DIR . ltrim( $rel_path, '/' );
    if ( file_exists( $path ) ) { return $require ? require_once $path : include $path; }
    return false;
}

// Core classes
cc_safe_include( 'includes/class-cc-options.php' );
cc_safe_include( 'includes/class-cc-fields.php' );
cc_safe_include( 'includes/class-cc-shortcode.php' );
cc_safe_include( 'admin/class-cc-admin.php' );

final class Custom_Checkout_Plugin {
    private static $instance = null;
    const SLUG = 'custom-checkout';
    const VER  = '2.5.0';

    public static function instance() { return self::$instance ?? ( self::$instance = new self() ); }
    private function __construct() { add_action( 'plugins_loaded', [ $this, 'maybe_init' ] ); }

    public function maybe_init() {
        if ( ! class_exists( 'WooCommerce' ) ) {
            add_action( 'admin_notices', function () {
                echo '<div class="notice notice-error"><p><strong>Custom Checkout</strong> requires WooCommerce.</p></div>';
            } );
            return;
        }

        // Admin UI
        if ( is_admin() && class_exists( 'CC_Admin' ) ) {
            add_action( 'admin_menu', [ 'CC_Admin', 'register' ] );
            add_action( 'admin_enqueue_scripts', [ 'CC_Admin', 'assets' ] );
        }
        // Shortcode (optional)
        if ( class_exists( 'CC_Shortcode' ) ) { add_action( 'init', [ 'CC_Shortcode', 'register' ] ); }

        // i18n
        add_action( 'init', function () { load_plugin_textdomain( 'custom-checkout', false, dirname( plugin_basename( __FILE__ ) ) . '/languages' ); } );

        // **THE FIX**: Use the correct WooCommerce hook that fires during the AJAX update.
        add_action( 'woocommerce_checkout_update_order_review', [ $this, 'sync_session_data_on_update' ] );

        // Front assets
        add_action( 'wp_enqueue_scripts', [ $this, 'enqueue_assets' ], 20 );

        // Override Woo template
        add_filter( 'woocommerce_locate_template', [ $this, 'load_templates' ], 10, 3 );

        // REST API
        add_action( 'rest_api_init', [ $this, 'register_routes' ] );

        // Checkout plumbing
        add_filter( 'woocommerce_checkout_posted_data', [ $this, 'merge_saved_data_into_checkout' ] );
        add_action( 'woocommerce_checkout_create_order', [ $this, 'save_custom_meta' ], 10, 2 );
        add_action( 'woocommerce_after_checkout_validation', [ $this, 'validate_required_fields' ], 10, 2 );
        add_action( 'woocommerce_thankyou', [ $this, 'clear_session_after_order' ] );
    }

    /**
     * Syncs our custom session data with the main WC_Customer object during the update_order_review AJAX call.
     * This ensures calculations (like coupons) have the correct customer data.
     */
    public function sync_session_data_on_update() {
        if ( WC()->session && ( $saved = WC()->session->get( 'cc_checkout' ) ) ) {
            $customer = WC()->customer;
            if ( ! $customer ) {
                return;
            }

            foreach ( $saved as $key => $value ) {
                if ( strpos( $key, 'billing_' ) === 0 || strpos( $key, 'shipping_' ) === 0 ) {
                    $setter = 'set_' . str_replace( 'cc_', '', $key );
                    if ( is_callable( [ $customer, $setter ] ) ) {
                        $customer->{$setter}( $value );
                    }
                }
            }
        }
    }

    public function enqueue_assets() {
        if ( ! is_checkout() || is_order_received_page() ) return;
        wp_enqueue_style( self::SLUG . '-css', CC_PLUGIN_URL . 'assets/css/checkout-style.css', [], self::VER );
        wp_enqueue_script( self::SLUG . '-js',  CC_PLUGIN_URL . 'assets/js/checkout-script.js', [ 'jquery','wc-checkout' ], self::VER, true );
        wp_enqueue_script( 'wc-address-i18n' );
        wp_enqueue_script( 'wc-country-select' );

        $opts = class_exists( 'CC_Options' ) ? CC_Options::get() : [];
        wp_localize_script( self::SLUG . '-js', 'CustomCheckout', [
            'nonce'    => wp_create_nonce( 'wp_rest' ),
            'ajax_url' => admin_url( 'admin-ajax.php' ),
            'restUrl'  => esc_url_raw( rest_url( 'custom-checkout/v1' ) ),
            'i18n'     => [
                'billing_required'  => __( 'Please complete required billing fields.', 'custom-checkout' ),
                'shipping_required' => __( 'Please complete required shipping address.', 'custom-checkout' ),
            ],
            'options' => [
                'regions' => [
                    'country'         => $opts['regions']['country'] ?? '',
                    'state'           => $opts['regions']['state'] ?? '',
                    'zips'            => $opts['regions']['zips'] ?? [],
                    'counties'        => $opts['regions']['counties'] ?? [],
                    'county_required' => ! empty( $opts['regions']['county_required'] ),
                ],
                'ui' => [
                    'hide_coupon' => ! empty( $opts['ui']['hide_coupon'] ),
                ],
            ],
        ] );

        if ( ! empty( $opts['ui']['hide_coupon'] ) ) {
            $custom_css = '.woocommerce-form-coupon-toggle { display: none !important; }';
            wp_add_inline_style( self::SLUG . '-css', $custom_css );
        }
    }

    public function load_templates( $template, $template_name, $template_path ) {
        if ( $template_name !== 'checkout/form-checkout.php' ) return $template;
        $enabled = true;
        if ( class_exists( 'CC_Options' ) ) { $opts = CC_Options::get(); $enabled = isset($opts['override_woo_checkout']) ? (bool)$opts['override_woo_checkout'] : true; }
        if ( ! $enabled ) return $template;
        $candidate = CC_PLUGIN_DIR . 'templates/woocommerce/checkout/form-checkout.php';
        return file_exists( $candidate ) ? $candidate : $template;
    }

    public function register_routes() {
        register_rest_route( 'custom-checkout/v1', '/save', [
            'methods'  => 'POST',
            'callback' => [ $this, 'rest_save' ],
            'permission_callback' => function () { $nonce = $_SERVER['HTTP_X_WP_NONCE'] ?? ''; return (bool) wp_verify_nonce( $nonce, 'wp_rest' ); },
            'args' => [ 'section' => [ 'required' => true, 'type' => 'string', 'enum' => [ 'billing','shipping','mode','extra' ] ], 'data' => [ 'required' => true, 'type' => 'object' ], ],
        ] );
        register_rest_route( 'custom-checkout/v1', '/state', [
            'methods'  => 'GET',
            'callback' => function () {
                if ( function_exists( 'wc_maybe_define_constant' ) ) wc_maybe_define_constant( 'WOOCOMMERCE_CART', true );
                if ( function_exists( 'wc_load_cart' ) ) wc_load_cart();
                if ( ! WC()->session ) { return new WP_REST_Response( [ 'ok' => false, 'message' => 'no-session' ], 500 ); }
                $saved = (array) WC()->session->get( 'cc_checkout', [] );
                return new WP_REST_Response( [ 'ok' => true, 'saved' => $saved ], 200 );
            },
            'permission_callback' => '__return_true',
        ] );
    }

    public function rest_save( $req ) {
        if ( function_exists( 'wc_maybe_define_constant' ) ) wc_maybe_define_constant( 'WOOCOMMERCE_CART', true );
        if ( function_exists( 'wc_load_cart' ) ) wc_load_cart();
        if ( ! WC()->session ) { return new WP_REST_Response( [ 'ok' => false, 'message' => 'No Woo session' ], 500 ); }

        $section = is_object($req) && method_exists($req,'get_param') ? $req->get_param('section') : '';
        $data    = is_object($req) && method_exists($req,'get_param') ? (array) $req->get_param('data') : [];

        $clean = [];
        switch ( $section ) {
            case 'billing': foreach ( $data as $k => $v ) { $clean[ 'billing_' . sanitize_key($k) ] = wc_clean( wp_unslash( $v ) ); } break;
            case 'shipping': foreach ( $data as $k => $v ) { $clean[ 'shipping_' . sanitize_key($k) ] = wc_clean( wp_unslash( $v ) ); } if ( isset( $data['order_comments'] ) ) { $clean['order_comments'] = wc_clean( wp_unslash( $data['order_comments'] ) ); } break;
            case 'extra': foreach ( $data as $k => $v ) { $clean[ 'extra_' . sanitize_key($k) ] = wc_clean( wp_unslash( $v ) ); } break;
            case 'mode': $mode = isset( $data['fulfillment_mode'] ) && in_array( $data['fulfillment_mode'], [ 'pickup', 'delivery' ], true ) ? $data['fulfillment_mode'] : 'pickup'; $clean['cc_fulfillment_mode'] = $mode; break;
        }

        $saved = (array) WC()->session->get( 'cc_checkout', [] );
        $saved = array_merge( $saved, $clean );
        WC()->session->set( 'cc_checkout', $saved );

        $customer = WC()->customer;
        if ( $customer && is_a( $customer, 'WC_Customer' ) ) {
            foreach ( $saved as $key => $val ) {
                if ( strpos( $key, 'billing_' ) === 0 || strpos( $key, 'shipping_' ) === 0 ) {
                    $method = 'set_' . $key; if ( is_callable( [ $customer, $method ] ) ) { $customer->$method( $val ); }
                }
            }
            $customer->save();
        }

        return new WP_REST_Response( [ 'ok' => true, 'saved' => $saved ], 200 );
    }

    public function merge_saved_data_into_checkout( $posted ) {
        if ( ! WC()->session ) return $posted;
        $saved = (array) WC()->session->get( 'cc_checkout', [] ); if ( empty( $saved ) ) return $posted;
        foreach ( $saved as $key => $val ) { if ( $val === '' || $val === null ) continue; $posted[ $key ] = $val; }
        if ( isset( $saved['cc_fulfillment_mode'] ) && $saved['cc_fulfillment_mode'] === 'pickup' ) { $posted['ship_to_different_address'] = 0; }
        else { $has_shipping = false; foreach ( [ 'first_name','last_name','address_1','city','postcode','country' ] as $k ) { if ( ! empty( $saved[ "shipping_{$k}" ] ) ) { $has_shipping = true; break; } } $posted['ship_to_different_address'] = $has_shipping ? 1 : 0; }
        return $posted;
    }

    public function validate_required_fields( $data, $errors ) {
        if ( ! WC()->session ) return;
        $saved = (array) WC()->session->get( 'cc_checkout', [] );
        $mode  = $saved['cc_fulfillment_mode'] ?? 'pickup';
        $opts  = class_exists('CC_Options') ? CC_Options::get() : [];

        $req_billing  = [];$req_shipping = [];
        foreach ( (array)($opts['fields']['billing'] ?? []) as $row ) { if ( ! empty($row['enabled']) && ! empty($row['required']) ) $req_billing[] = $row['key']; }
        foreach ( (array)($opts['fields']['shipping'] ?? []) as $row ) { if ( ! empty($row['enabled']) && ! empty($row['required']) ) $req_shipping[] = $row['key']; }

        foreach ( $req_billing as $k ) { $val = $saved['billing_' . $k] ?? ''; if ( $val === '' ) $errors->add( 'validation', sprintf( __( 'Billing %s is required.', 'custom-checkout' ), esc_html( str_replace('_',' ', $k) ) ) ); }
        if ( ! empty( $saved['billing_email'] ) && ! is_email( $saved['billing_email'] ) ) { $errors->add( 'validation', __( 'Please enter a valid billing email address.', 'custom-checkout' ) ); }
        if ( $mode === 'delivery' ) { foreach ( $req_shipping as $k ) { $val = $saved['shipping_' . $k] ?? ''; if ( $val === '' ) $errors->add( 'validation', sprintf( __( 'Shipping %s is required for delivery.', 'custom-checkout' ), esc_html( str_replace('_',' ', $k) ) ) ); } }

        $countries = new WC_Countries();
        $zip_whitelist = isset( $opts['regions']['zips'] ) && is_array( $opts['regions']['zips'] ) ? $opts['regions']['zips'] : [];
        $limit_country = isset( $opts['regions']['country'] ) ? strtoupper( $opts['regions']['country'] ) : '';
        $limit_state   = isset( $opts['regions']['state'] ) ? strtoupper( $opts['regions']['state'] ) : '';
        $counties      = isset( $opts['regions']['counties'] ) && is_array( $opts['regions']['counties'] ) ? array_map( 'strtoupper', array_map( 'trim', $opts['regions']['counties'] ) ) : [];
        $county_req    = ! empty( $opts['regions']['county_required'] );

        foreach ( [ 'billing', 'shipping' ] as $scope ) {
            $country = strtoupper( $saved["{$scope}_country"] ?? '' );
            $state   = strtoupper( $saved["{$scope}_state"] ?? '' );
            $county  = strtoupper( trim( $saved["{$scope}_county"] ?? '' ) );

            if ( $limit_country && $country && $country !== $limit_country ) $errors->add( 'validation', __( 'Selected country is not allowed for checkout.', 'custom-checkout' ) );
            if ( $limit_state && $state && $state !== $limit_state ) $errors->add( 'validation', __( 'Selected state is not allowed for checkout.', 'custom-checkout' ) );

            if ( $country ) {
                $all_countries = array_keys( $countries->get_countries() );
                if ( ! in_array( $country, $all_countries, true ) ) $errors->add( 'validation', sprintf( __( '%s country must be a valid 2‑letter code.', 'custom-checkout' ), ucfirst( $scope ) ) );
                $states = $countries->get_states( $country );
                if ( is_array( $states ) && ! empty( $states ) && $state && ! array_key_exists( $state, $states ) ) $errors->add( 'validation', sprintf( __( '%s state is not valid for the selected country.', 'custom-checkout' ), ucfirst( $scope ) ) );
            }

            if ( $county_req && ! empty( $counties ) ) {
                if ( $scope === 'billing' && $county === '' ) $errors->add( 'validation', __( 'Billing county is required.', 'custom-checkout' ) );
                if ( $scope === 'shipping' && $mode === 'delivery' && $county === '' ) $errors->add( 'validation', __( 'Shipping county is required for delivery.', 'custom-checkout' ) );
                if ( $county && ! in_array( $county, $counties, true ) ) $errors->add( 'validation', __( 'Selected county is not allowed.', 'custom-checkout' ) );
            }
        }

        if ( $mode === 'delivery' && ! empty( $zip_whitelist ) ) {
            $postcode = strtoupper( trim( $saved['shipping_postcode'] ?? '' ) );
            $normalized = array_map( 'strtoupper', array_map( 'trim', $zip_whitelist ) );
            if ( $postcode && ! in_array( $postcode, $normalized, true ) ) $errors->add( 'validation', __( 'We currently deliver only to selected ZIP codes. Please check your postcode or choose pickup.', 'custom-checkout' ) );
        }
    }

    public function save_custom_meta( $order, $data ) {
        if ( ! WC()->session ) return; $saved = (array) WC()->session->get( 'cc_checkout', [] );
        if ( isset( $saved['cc_fulfillment_mode'] ) ) { $order->update_meta_data( '_cc_fulfillment_mode', $saved['cc_fulfillment_mode'] ); }
        foreach ( $saved as $k => $v ) { if ( strpos( $k, 'extra_' ) === 0 ) { $order->update_meta_data( '_cc_' . substr( $k, 6 ), $v ); } }
    }

    public function clear_session_after_order( $order_id ) { if ( WC()->session ) { WC()->session->__unset( 'cc_checkout' ); } }
}
Custom_Checkout_Plugin::instance();