<?php
if ( ! defined( 'ABSPATH' ) ) exit;

class CC_Admin {
    const MENU_SLUG = 'custom-checkout';
    const OPT_FALLBACK = 'custom_checkout_options';

    /** Register admin page */
    public static function register() {
        add_menu_page(
                __( 'Custom Checkout', 'custom-checkout' ),
                __( 'Custom Checkout', 'custom-checkout' ),
                'manage_woocommerce',
                self::MENU_SLUG,
                [ __CLASS__, 'render_page' ],
                'dashicons-cart',
                56
        );
    }

    /** Enqueue admin assets (CSS/JS) and expose img path */
    public static function assets( $hook = '' ) {
        // Only on our page
        if ( strpos( $hook, 'toplevel_page_' . self::MENU_SLUG ) === false ) return;

        wp_enqueue_style( 'cc-admin', CC_PLUGIN_URL . 'assets/admin/admin.css', [], Custom_Checkout_Plugin::VER );
        wp_enqueue_script( 'cc-admin', CC_PLUGIN_URL . 'assets/admin/admin.js', [ 'jquery' ], Custom_Checkout_Plugin::VER, true );
        wp_localize_script( 'cc-admin', 'CCAdmin', [
                'imgUrl' => CC_PLUGIN_URL . 'assets/img',
        ] );
    }

    /** Get options via CC_Options if present, otherwise wp_options fallback */
    private static function get_opts() {
        if ( class_exists( 'CC_Options' ) && is_callable( [ 'CC_Options', 'get' ] ) ) {
            $opts = CC_Options::get();
        } else {
            $opts = get_option( self::OPT_FALLBACK, [] );
        }
        return is_array( $opts ) ? $opts : [];
    }

    /** Persist options via CC_Options if present; fallback to wp_options */
    private static function update_opts( $opts ) {
        if ( class_exists( 'CC_Options' ) && is_callable( [ 'CC_Options', 'update' ] ) ) {
            CC_Options::update( $opts );
        } else {
            update_option( self::OPT_FALLBACK, $opts );
        }
    }

    /** Sanitize helpers */
    private static function sanitize_text_arr( $arr ) {
        $out = [];
        foreach ( (array) $arr as $k => $v ) {
            $out[ sanitize_key( $k ) ] = is_array( $v ) ? self::sanitize_text_arr( $v ) : sanitize_text_field( wp_unslash( $v ) );
        }
        return $out;
    }

    private static function sanitize_bool( $v ) { return ! empty( $v ) ? 1 : 0; }

    /** Render admin page */
    public static function render_page() {
        if ( ! current_user_can( 'manage_woocommerce' ) ) return;

        $opts = self::get_opts();

        // Save
        if ( isset( $_POST['cc_admin_save'] ) && check_admin_referer( 'cc_admin_save_action', 'cc_admin_nonce' ) ) {
            // UI
            $opts['ui'] = [
                    'hide_coupon' => self::sanitize_bool( $_POST['ui']['hide_coupon'] ?? 0 ),
            ];
            // Regions
            $opts['regions'] = [
                    'country'         => strtoupper( sanitize_text_field( wp_unslash( $_POST['regions']['country'] ?? '' ) ) ),
                    'state'           => strtoupper( sanitize_text_field( wp_unslash( $_POST['regions']['state'] ?? '' ) ) ),
                    'zips'            => array_filter( array_map( 'trim', preg_split( '/[\r\n,]+/', (string) ( $_POST['regions']['zips'] ?? '' ) ) ) ),
                    'counties'        => array_filter( array_map( 'trim', preg_split( '/[\r\n,]+/', (string) ( $_POST['regions']['counties'] ?? '' ) ) ) ),
                    'county_required' => self::sanitize_bool( $_POST['regions']['county_required'] ?? 0 ),
            ];
            // Fields
            $fields_in = $_POST['fields'] ?? [];
            $sections  = [ 'billing', 'shipping', 'extra' ];
            $opts['fields'] = [];
            foreach ( $sections as $section ) {
                $rows = (array) ( $fields_in[ $section ] ?? [] );
                $clean_rows = [];
                foreach ( $rows as $row ) {
                    $label = sanitize_text_field( wp_unslash( $row['label'] ?? '' ) );
                    $key   = sanitize_key( $row['key'] ?? '' );
                    if ( $label === '' || $key === '' ) continue;
                    $type  = in_array( $row['type'] ?? 'text', [ 'text','email','tel','textarea','select','checkbox','date','country' ], true ) ? $row['type'] : 'text';
                    $required = self::sanitize_bool( $row['required'] ?? 0 );
                    $enabled  = self::sanitize_bool( $row['enabled'] ?? 0 );
                    $width = in_array( $row['width'] ?? 'full', [ 'full','1/2','1/3' ], true ) ? $row['width'] : 'full';
                    $clean_rows[] = compact( 'label','key','type','required','enabled','width' );
                }
                $opts['fields'][ $section ] = $clean_rows;
            }

            self::update_opts( $opts );
            echo '<div class="updated"><p>' . esc_html__( 'Settings saved.', 'custom-checkout' ) . '</p></div>';
        }

        // Defaults for empty state
        $opts += [
                'ui' => [ 'hide_coupon' => 0 ],
                'regions' => [ 'country' => '', 'state' => '', 'zips' => [], 'counties' => [], 'county_required' => 0 ],
                'fields' => [ 'billing' => [], 'shipping' => [], 'extra' => [] ],
        ];

        ?>
        <div class="wrap cc-admin">
            <h1><?php esc_html_e( 'Custom Checkout', 'custom-checkout' ); ?></h1>

            <nav class="cc-tabs">
                <a href="#" class="cc-tab is-active" data-tab="general"><?php esc_html_e( 'General', 'custom-checkout' ); ?></a>
                <a href="#" class="cc-tab" data-tab="regions"><?php esc_html_e( 'Regions & Delivery', 'custom-checkout' ); ?></a>
                <a href="#" class="cc-tab" data-tab="fields"><?php esc_html_e( 'Field Builder', 'custom-checkout' ); ?></a>
            </nav>

            <form method="post" action="">
                <?php wp_nonce_field( 'cc_admin_save_action', 'cc_admin_nonce' ); ?>

                <section class="cc-panel is-active" data-panel="general">
                    <div class="cc-card">
                        <h2><?php esc_html_e( 'Checkout UI', 'custom-checkout' ); ?></h2>
                        <label>
                            <input type="checkbox" name="ui[hide_coupon]" value="1" <?php checked( ! empty( $opts['ui']['hide_coupon'] ) ); ?> />
                            <?php esc_html_e( 'Hide Woo coupon on checkout (coupon stays on Cart)', 'custom-checkout' ); ?>
                        </label>
                        <p class="cc-muted"><?php esc_html_e( 'If enabled, the coupon prompt and form are hidden on the checkout page.', 'custom-checkout' ); ?></p>
                    </div>
                </section>

                <section class="cc-panel" data-panel="regions">
                    <div class="cc-card">
                        <h2><?php esc_html_e( 'Regions & Delivery Limits', 'custom-checkout' ); ?></h2>
                        <table class="form-table">
                            <tr>
                                <th><label for="cc-country"><?php esc_html_e( 'Limit Country (2‑letter code)', 'custom-checkout' ); ?></label></th>
                                <td><input id="cc-country" type="text" name="regions[country]" value="<?php echo esc_attr( $opts['regions']['country'] ); ?>" class="regular-text" placeholder="US" /></td>
                            </tr>
                            <tr>
                                <th><label for="cc-state"><?php esc_html_e( 'Limit State/Province (2‑letter code)', 'custom-checkout' ); ?></label></th>
                                <td><input id="cc-state" type="text" name="regions[state]" value="<?php echo esc_attr( $opts['regions']['state'] ); ?>" class="regular-text" placeholder="CA" /></td>
                            </tr>
                            <tr>
                                <th><label for="cc-zips"><?php esc_html_e( 'Allowed ZIP Codes (one per line or comma‑separated)', 'custom-checkout' ); ?></label></th>
                                <td><textarea id="cc-zips" name="regions[zips]" rows="4" class="large-text"><?php echo esc_textarea( implode( "\n", (array) $opts['regions']['zips'] ) ); ?></textarea></td>
                            </tr>
                            <tr>
                                <th><label for="cc-counties"><?php esc_html_e( 'Allowed Counties (one per line or comma‑separated)', 'custom-checkout' ); ?></label></th>
                                <td><textarea id="cc-counties" name="regions[counties]" rows="4" class="large-text"><?php echo esc_textarea( implode( "\n", (array) $opts['regions']['counties'] ) ); ?></textarea></td>
                            </tr>
                            <tr>
                                <th><?php esc_html_e( 'County required?', 'custom-checkout' ); ?></th>
                                <td><label><input type="checkbox" name="regions[county_required]" value="1" <?php checked( ! empty( $opts['regions']['county_required'] ) ); ?> /> <?php esc_html_e( 'Require county on billing (and shipping if delivery).', 'custom-checkout' ); ?></label></td>
                            </tr>
                        </table>
                    </div>
                </section>

                <section class="cc-panel" data-panel="fields">
                    <div class="cc-card">
                        <h2><?php esc_html_e( 'Field Builder', 'custom-checkout' ); ?></h2>
                        <p class="cc-muted"><?php esc_html_e( 'Add, remove, enable/disable, and set required flags. Keys map to Woo fields (e.g., first_name) or become extra meta (in the “extra” section).', 'custom-checkout' ); ?></p>

                        <?php foreach ( [ 'billing' => __( 'Billing Fields', 'custom-checkout' ), 'shipping' => __( 'Shipping Fields', 'custom-checkout' ), 'extra' => __( 'Extra Fields', 'custom-checkout' ) ] as $section => $label ) : ?>
                            <h3><?php echo esc_html( $label ); ?></h3>
                            <table class="cc-table" data-section="<?php echo esc_attr( $section ); ?>">
                                <thead>
                                <tr>
                                    <th class="cc-col-order"><?php esc_html_e( 'Order', 'custom-checkout' ); ?></th>
                                    <th class="cc-col-label"><?php esc_html_e( 'Label', 'custom-checkout' ); ?></th>
                                    <th class="cc-col-key"><?php esc_html_e( 'Key', 'custom-checkout' ); ?></th>
                                    <th class="cc-col-type"><?php esc_html_e( 'Type', 'custom-checkout' ); ?></th>
                                    <th class="cc-col-req"><?php esc_html_e( 'Required', 'custom-checkout' ); ?></th>
                                    <th class="cc-col-en"><?php esc_html_e( 'Enabled', 'custom-checkout' ); ?></th>
                                    <th class="cc-col-width"><?php esc_html_e( 'Width', 'custom-checkout' ); ?></th>
                                    <th class="cc-col-actions"><?php esc_html_e( 'Actions', 'custom-checkout' ); ?></th>
                                </tr>
                                </thead>
                                <tbody>
                                <?php $rows = (array) ( $opts['fields'][ $section ] ?? [] ); ?>
                                <?php foreach ( $rows as $i => $row ) :
                                    $label_v = $row['label'] ?? '';
                                    $key_v   = $row['key'] ?? '';
                                    $type_v  = $row['type'] ?? 'text';
                                    $req_v   = ! empty( $row['required'] );
                                    $en_v    = ! empty( $row['enabled'] );
                                    $width_v = $row['width'] ?? 'full';
                                    ?>
                                    <tr>
                                        <td class="cc-order cc-col-order">
                                            <button type="button" class="cc-up" aria-label="Move up"><img src="<?php echo esc_url( CC_PLUGIN_URL . 'assets/img/chevron-up.svg' ); ?>" width="16" height="16" alt="" aria-hidden="true" /></button>
                                            <button type="button" class="cc-down" aria-label="Move down"><img src="<?php echo esc_url( CC_PLUGIN_URL . 'assets/img/chevron-down.svg' ); ?>" width="16" height="16" alt="" aria-hidden="true" /></button>
                                        </td>
                                        <td class="cc-col-label"><input type="text" name="fields[<?php echo esc_attr( $section ); ?>][<?php echo esc_attr( $i ); ?>][label]" value="<?php echo esc_attr( $label_v ); ?>" /></td>
                                        <td class="cc-col-key"><input type="text" name="fields[<?php echo esc_attr( $section ); ?>][<?php echo esc_attr( $i ); ?>][key]" value="<?php echo esc_attr( $key_v ); ?>" /></td>
                                        <td class="cc-col-type">
                                            <select name="fields[<?php echo esc_attr( $section ); ?>][<?php echo esc_attr( $i ); ?>][type]">
                                                <?php foreach ( [ 'text','email','tel','textarea','select','checkbox','date','country' ] as $t ) : ?>
                                                    <option value="<?php echo esc_attr( $t ); ?>" <?php selected( $type_v, $t ); ?>><?php echo esc_html( $t ); ?></option>
                                                <?php endforeach; ?>
                                            </select>
                                        </td>
                                        <td class="cc-center cc-col-req"><input type="checkbox" name="fields[<?php echo esc_attr( $section ); ?>][<?php echo esc_attr( $i ); ?>][required]" value="1" <?php checked( $req_v ); ?> /></td>
                                        <td class="cc-center cc-col-en"><input type="checkbox" name="fields[<?php echo esc_attr( $section ); ?>][<?php echo esc_attr( $i ); ?>][enabled]" value="1" <?php checked( $en_v ); ?> /></td>
                                        <td class="cc-col-width">
                                            <select name="fields[<?php echo esc_attr( $section ); ?>][<?php echo esc_attr( $i ); ?>][width]">
                                                <?php foreach ( [ 'full','1/2','1/3' ] as $w ) : ?>
                                                    <option value="<?php echo esc_attr( $w ); ?>" <?php selected( $width_v, $w ); ?>><?php echo esc_html( $w ); ?></option>
                                                <?php endforeach; ?>
                                            </select>
                                        </td>
                                        <td class="cc-col-actions"><button type="button" class="cc-remove" aria-label="Remove"><img src="<?php echo esc_url( CC_PLUGIN_URL . 'assets/img/x.svg' ); ?>" width="16" height="16" alt="" aria-hidden="true" /></button></td>
                                    </tr>
                                <?php endforeach; ?>
                                </tbody>
                            </table>
                            <p><button type="button" class="button cc-add" data-section="<?php echo esc_attr( $section ); ?>"><?php esc_html_e( 'Add field', 'custom-checkout' ); ?></button></p>
                        <?php endforeach; ?>
                    </div>
                </section>

                <p class="submit"><button type="submit" name="cc_admin_save" class="button button-primary"><?php esc_html_e( 'Save changes', 'custom-checkout' ); ?></button></p>
            </form>
        </div>
        <?php
    }
}