<?php
if ( ! defined( 'ABSPATH' ) ) exit;

class CC_Options {
    const OPTION_KEY = 'cc_options';

    /** Default full options tree */
    public static function defaults() {
        return [
            'override_woo_checkout' => true,
            'regions' => [
                'country'         => '', // e.g. US
                'state'           => '', // e.g. CA
                'zips'            => [], // whitelist for delivery
                'counties'        => [], // optional list of allowed counties
                'county_required' => false,
            ],
            'prefill' => [
                'email_from_user' => true,
            ],
            'ui' => [
                'delivery_note' => 'For delivery, well collect your address and call you with the delivery price (cost varies by address).',
            ],
            // Phase 2  field builder data model (safe defaults)
            'fields' => [
                'billing'  => self::default_billing_fields(),
                'shipping' => self::default_shipping_fields(),
                'extra'    => [],
            ],
        ];
    }

    public static function default_billing_fields() {
        return [
            ['key'=>'first_name','label'=>'First name','type'=>'text','required'=>true,'enabled'=>true,'width'=>'1/2'],
            ['key'=>'last_name', 'label'=>'Last name','type'=>'text','required'=>true,'enabled'=>true,'width'=>'1/2'],
            ['key'=>'email',     'label'=>'Email','type'=>'email','required'=>true,'enabled'=>true,'width'=>'1/2'],
            ['key'=>'phone',     'label'=>'Phone','type'=>'tel','required'=>false,'enabled'=>true,'width'=>'1/2'],
            ['key'=>'company',   'label'=>'Company','type'=>'text','required'=>false,'enabled'=>true,'width'=>'full'],
            ['key'=>'address_1', 'label'=>'Address line 1','type'=>'text','required'=>true,'enabled'=>true,'width'=>'full'],
            ['key'=>'address_2', 'label'=>'Address line 2','type'=>'text','required'=>false,'enabled'=>true,'width'=>'full'],
            ['key'=>'city',      'label'=>'City','type'=>'text','required'=>true,'enabled'=>true,'width'=>'1/3'],
            ['key'=>'state',     'label'=>'State/Province','type'=>'text','required'=>false,'enabled'=>true,'width'=>'1/3'],
            ['key'=>'postcode',  'label'=>'Postcode','type'=>'text','required'=>true,'enabled'=>true,'width'=>'1/3'],
            ['key'=>'country',   'label'=>'Country','type'=>'country','required'=>true,'enabled'=>true,'width'=>'full'],
        ];
    }

    public static function default_shipping_fields() {
        return [
            ['key'=>'first_name','label'=>'First name','type'=>'text','required'=>true,'enabled'=>true,'width'=>'1/2'],
            ['key'=>'last_name', 'label'=>'Last name','type'=>'text','required'=>true,'enabled'=>true,'width'=>'1/2'],
            ['key'=>'company',   'label'=>'Company','type'=>'text','required'=>false,'enabled'=>true,'width'=>'full'],
            ['key'=>'address_1', 'label'=>'Address line 1','type'=>'text','required'=>true,'enabled'=>true,'width'=>'full'],
            ['key'=>'address_2', 'label'=>'Address line 2','type'=>'text','required'=>false,'enabled'=>true,'width'=>'full'],
            ['key'=>'city',      'label'=>'City','type'=>'text','required'=>true,'enabled'=>true,'width'=>'1/3'],
            ['key'=>'state',     'label'=>'State/Province','type'=>'text','required'=>true,'enabled'=>true,'width'=>'1/3'],
            ['key'=>'postcode',  'label'=>'Postcode','type'=>'text','required'=>true,'enabled'=>true,'width'=>'1/3'],
            ['key'=>'country',   'label'=>'Country','type'=>'country','required'=>true,'enabled'=>true,'width'=>'full'],
        ];
    }

    /**
     * Get options merged with defaults safely (deep merge) and normalize lists.
     */
    public static function get() {
        $def = self::defaults();
        $opt = get_option( self::OPTION_KEY, [] );
        if ( ! is_array( $opt ) ) { $opt = []; }

        // Deep merge
        $merged = array_replace_recursive( $def, $opt );

        // Ensure required nested arrays exist (avoid undefined-index notices)
        foreach ( ['regions','prefill','ui','fields'] as $root ) {
            if ( ! isset( $merged[ $root ] ) || ! is_array( $merged[ $root ] ) ) {
                $merged[ $root ] = $def[ $root ];
            }
        }
        foreach ( ['zips','counties'] as $k ) {
            if ( ! isset( $merged['regions'][ $k ] ) ) {
                $merged['regions'][ $k ] = [];
            }
            // Accept comma/space/line separated strings and turn into arrays
            if ( is_string( $merged['regions'][ $k ] ) ) {
                $merged['regions'][ $k ] = self::parse_list( $merged['regions'][ $k ] );
            }
            if ( ! is_array( $merged['regions'][ $k ] ) ) {
                $merged['regions'][ $k ] = [];
            }
        }
        if ( ! isset( $merged['regions']['county_required'] ) ) {
            $merged['regions']['county_required'] = false;
        }

        // Fields groups safety
        foreach ( ['billing','shipping','extra'] as $sec ) {
            if ( ! isset( $merged['fields'][ $sec ] ) || ! is_array( $merged['fields'][ $sec ] ) ) {
                $merged['fields'][ $sec ] = $def['fields'][ $sec ];
            }
        }

        return $merged;
    }

    /** Convert comma/space/line separated list to unique array */
    public static function parse_list( $str ) {
        $parts = preg_split( '/[\s,]+/', (string) $str );
        $parts = array_filter( array_map( 'trim', $parts ), function( $z ) { return $z !== ''; } );
        return array_values( array_unique( $parts ) );
    }

    /** Save options safely from $_POST-like structure */
    public static function save( $data ) {
        $def = self::defaults();
        $out = $def; // start with defaults

        // General flags
        $out['override_woo_checkout'] = ! empty( $data['override_woo_checkout'] );

        // Regions
        $out['regions']['country'] = isset( $data['regions']['country'] ) ? strtoupper( sanitize_text_field( $data['regions']['country'] ) ) : '';
        $out['regions']['state']   = isset( $data['regions']['state'] )   ? strtoupper( sanitize_text_field( $data['regions']['state'] ) )     : '';

        foreach ( ['zips','counties'] as $k ) {
            if ( isset( $data['regions'][ $k ] ) ) {
                $out['regions'][ $k ] = is_array( $data['regions'][ $k ] )
                    ? array_map( 'sanitize_text_field', $data['regions'][ $k ] )
                    : self::parse_list( $data['regions'][ $k ] );
            } else {
                $out['regions'][ $k ] = [];
            }
        }
        $out['regions']['county_required'] = ! empty( $data['regions']['county_required'] );

        // Prefill & UI
        $out['prefill']['email_from_user'] = ! empty( $data['prefill']['email_from_user'] );
        $out['ui']['delivery_note'] = isset( $data['ui']['delivery_note'] )
            ? wp_kses_post( $data['ui']['delivery_note'] )
            : $def['ui']['delivery_note'];

        // Field Builder (Phase 2 skeleton)
        if ( isset( $data['fields'] ) && is_array( $data['fields'] ) ) {
            $out['fields']['billing']  = self::sanitize_fields( $data['fields']['billing']  ?? [], 'billing' );
            $out['fields']['shipping'] = self::sanitize_fields( $data['fields']['shipping'] ?? [], 'shipping' );
            $out['fields']['extra']    = self::sanitize_fields( $data['fields']['extra']    ?? [], 'extra' );
        }

        update_option( self::OPTION_KEY, $out );
        return $out;
    }

    /** Sanitize a field list row-by-row */
    public static function sanitize_fields( $arr, $section ) {
        $allowed_types = ['text','email','tel','textarea','select','checkbox','date','country'];
        $clean = [];
        if ( ! is_array( $arr ) ) return $clean;
        foreach ( $arr as $row ) {
            $key = isset($row['key']) ? sanitize_key( $row['key'] ) : '';
            if ( $key === '' ) continue;
            $clean[] = [
                'key'      => $key,
                'label'    => isset($row['label']) ? sanitize_text_field($row['label']) : $key,
                'type'     => in_array($row['type'] ?? '', $allowed_types, true) ? $row['type'] : 'text',
                'required' => ! empty( $row['required'] ),
                'enabled'  => ! empty( $row['enabled'] ),
                'width'    => in_array($row['width'] ?? 'full', ['full','1/2','1/3'], true) ? $row['width'] : 'full',
            ];
        }
        return $clean;
    }
}