<?php
/**
 * Dynamic checkout template rendered from Field Builder config
 */
defined( 'ABSPATH' ) || exit;

do_action( 'woocommerce_before_checkout_form', $checkout );

if ( ! $checkout->is_registration_enabled() && $checkout->is_registration_required() && ! is_user_logged_in() ) {
    echo esc_html( apply_filters( 'woocommerce_checkout_must_be_logged_in_message', __( 'You must be logged in to checkout.', 'woocommerce' ) ) );
    return;
}

$countries_obj  = new WC_Countries();
$all_countries  = $countries_obj->get_countries();
$us_states      = $countries_obj->get_states( 'US' );
$opts           = class_exists('CC_Options') ? CC_Options::get() : [];
$fields         = isset( $opts['fields'] ) ? $opts['fields'] : [ 'billing'=>[], 'shipping'=>[], 'extra'=>[] ];
$limit_country  = isset($opts['regions']['country']) ? strtoupper($opts['regions']['country']) : '';
$limit_state    = isset($opts['regions']['state']) ? strtoupper($opts['regions']['state']) : '';
$counties       = isset($opts['regions']['counties']) && is_array($opts['regions']['counties']) ? $opts['regions']['counties'] : [];
$county_req     = ! empty($opts['regions']['county_required']);
$delivery_note  = isset($opts['ui']['delivery_note']) ? $opts['ui']['delivery_note'] : '';

$prefill_email  = '';
if ( ! empty($opts['prefill']['email_from_user']) && is_user_logged_in() ) {
    $user = wp_get_current_user();
    if ( $user && $user->user_email ) { $prefill_email = $user->user_email; }
}

/**
 * Render a single field from config.
 */
function cc_render_field( $section, $f, $ctx ) {
    $key      = $f['key'];
    $label    = $f['label'];
    $type     = $f['type'];
    $required = ! empty( $f['required'] );
    $enabled  = ! empty( $f['enabled'] );

    if ( ! $enabled ) return '';

    $id = 'cc_' . esc_attr( $section . '_' . $key );
    $req = $required ? ' required' : '';

    // Prefill email for billing if requested
    $prefill_email = $ctx['prefill_email'];

    ob_start();
    echo '<p><label>' . esc_html( $label );

    // Special cases: country/state/county
    if ( $key === 'country' || $type === 'country' ) {
        // If admin limited country, render a single-option select so UI stays uniform
        if ( $ctx['limit_country'] ) {
            $code = $ctx['limit_country'];
            echo ' <select id="' . esc_attr( $id ) . '" data-section="' . esc_attr( $section ) . '" data-key="country" autocomplete="' . esc_attr( $section ) . ' country"' . $req . '>';
            echo '<option value="' . esc_attr( $code ) . '">' . esc_html( $ctx['all_countries'][ $code ] ?? $code ) . '</option>';
            echo '</select>';
        } else {
            echo ' <select id="' . esc_attr( $id ) . '" data-section="' . esc_attr( $section ) . '" data-key="country" autocomplete="' . esc_attr( $section ) . ' country"' . $req . '>';
            echo '<option value="">' . esc_html__( 'Select a country', 'custom-checkout' ) . '</option>';
            foreach ( $ctx['all_countries'] as $code => $name ) {
                echo '<option value="' . esc_attr( $code ) . '">' . esc_html( $name ) . '</option>';
            }
            echo '</select>';
        }
    }
    elseif ( $key === 'state' ) {
        // If country US use states list; if admin locked state render single option
        if ( $ctx['limit_country'] === 'US' ) {
            $disabled_attr = $ctx['limit_state'] ? '' : '';
            echo ' <select id="' . esc_attr( $id ) . '" data-section="' . esc_attr( $section ) . '" data-key="state" autocomplete="' . esc_attr( $section ) . ' address-level1"' . $req . '>';
            if ( ! $ctx['limit_state'] ) echo '<option value="">' . esc_html__( 'Select state', 'custom-checkout' ) . '</option>';
            foreach ( (array) $ctx['us_states'] as $code => $name ) {
                if ( $ctx['limit_state'] && strtoupper($code) !== $ctx['limit_state'] ) continue;
                echo '<option value="' . esc_attr( $code ) . '">' . esc_html( $name ) . '</option>';
            }
            echo '</select>';
        } else {
            echo ' <input type="text" id="' . esc_attr( $id ) . '" data-section="' . esc_attr( $section ) . '" data-key="state" autocomplete="' . esc_attr( $section ) . ' address-level1"' . $req . ' />';
        }
    }
    elseif ( $key === 'county' && ! empty( $ctx['counties'] ) ) {
        $req_attr = ( $ctx['county_req'] || $required ) ? ' required' : '';
        echo ' <select id="' . esc_attr( $id ) . '" data-section="' . esc_attr( $section ) . '" data-key="county"' . $req_attr . '>';
        echo '<option value="">' . esc_html__( 'Select county', 'custom-checkout' ) . '</option>';
        foreach ( $ctx['counties'] as $c ) {
            echo '<option value="' . esc_attr( $c ) . '">' . esc_html( $c ) . '</option>';
        }
        echo '</select>';
    }
    else {
        // Generic types
        switch ( $type ) {
            case 'email':
                $val = ( $section === 'billing' && $key === 'email' ) ? esc_attr( $prefill_email ) : '';
                echo ' <input type="email" id="' . esc_attr( $id ) . '" value="' . $val . '" data-section="' . esc_attr( $section ) . '" data-key="' . esc_attr( $key ) . '" autocomplete="' . esc_attr( $section ) . ' email"' . $req . ' />';
                break;
            case 'tel':
                echo ' <input type="tel" id="' . esc_attr( $id ) . '" data-section="' . esc_attr( $section ) . '" data-key="' . esc_attr( $key ) . '" autocomplete="' . esc_attr( $section ) . ' tel"' . $req . ' />';
                break;
            case 'textarea':
                echo ' <textarea id="' . esc_attr( $id ) . '" rows="3" data-section="' . esc_attr( $section ) . '" data-key="' . esc_attr( $key ) . '"></textarea>';
                break;
            default: // text
                $ac = 'text';
                if ( in_array( $key, [ 'address_1','address_2' ], true ) ) $ac = 'address-line' . ( $key === 'address_1' ? '1' : '2' );
                elseif ( $key === 'city' ) $ac = 'address-level2';
                elseif ( $key === 'postcode' ) $ac = 'postal-code';
                elseif ( $key === 'first_name' ) $ac = 'given-name';
                elseif ( $key === 'last_name' ) $ac = 'family-name';
                echo ' <input type="text" id="' . esc_attr( $id ) . '" data-section="' . esc_attr( $section ) . '" data-key="' . esc_attr( $key ) . '" autocomplete="' . esc_attr( $section . ' ' . $ac ) . '"' . $req . ' />';
        }
    }

    echo '</label></p>';
    return ob_get_clean();
}

/**
 * Render a section (billing/shipping/extra) honoring widths.
 */
function cc_render_section_fields( $section, $fields, $ctx ) {
    $half = [];$third = [];
    foreach ( $fields as $f ) {
        if ( empty( $f['enabled'] ) ) continue;
        $width = isset( $f['width'] ) ? $f['width'] : 'full';
        if ( $width === '1/2' )      $half[]  = $f;
        elseif ( $width === '1/3' )  $third[] = $f;
        else {
            // flush rows waiting
            if ( ! empty( $half ) ) {
                echo '<div class="cc-grid-2">';
                for ( $i=0; $i<count($half); $i+=2 ) {
                    echo cc_render_field( $section, $half[$i], $ctx );
                    if ( isset($half[$i+1]) ) echo cc_render_field( $section, $half[$i+1], $ctx );
                }
                echo '</div>';
                $half = [];
            }
            if ( ! empty( $third ) ) {
                echo '<div class="cc-grid-3">';
                for ( $i=0; $i<count($third); $i+=3 ) {
                    echo cc_render_field( $section, $third[$i], $ctx );
                    if ( isset($third[$i+1]) ) echo cc_render_field( $section, $third[$i+1], $ctx );
                    if ( isset($third[$i+2]) ) echo cc_render_field( $section, $third[$i+2], $ctx );
                }
                echo '</div>';
                $third = [];
            }
            // full width
            echo cc_render_field( $section, $f, $ctx );
        }
    }
    // tail flush
    if ( ! empty( $half ) ) {
        echo '<div class="cc-grid-2">';
        for ( $i=0; $i<count($half); $i+=2 ) {
            echo cc_render_field( $section, $half[$i], $ctx );
            if ( isset($half[$i+1]) ) echo cc_render_field( $section, $half[$i+1], $ctx );
        }
        echo '</div>';
    }
    if ( ! empty( $third ) ) {
        echo '<div class="cc-grid-3">';
        for ( $i=0; $i<count($third); $i+=3 ) {
            echo cc_render_field( $section, $third[$i], $ctx );
            if ( isset($third[$i+1]) ) echo cc_render_field( $section, $third[$i+1], $ctx );
            if ( isset($third[$i+2]) ) echo cc_render_field( $section, $third[$i+2], $ctx );
        }
        echo '</div>';
    }
}
?>

<div class="cc-wrap">
  <main class="cc-main">
    <nav class="cc-steps" role="tablist" aria-label="<?php esc_attr_e( 'Checkout Steps', 'custom-checkout' ); ?>">
      <button id="stepper-1" class="cc-stepper-btn step-active" type="button" role="tab" aria-selected="true" aria-controls="step-1"><?php esc_html_e( 'Billing', 'custom-checkout' ); ?></button>
      <button id="stepper-2" class="cc-stepper-btn step-inactive" type="button" role="tab" aria-selected="false" aria-controls="step-2"><?php esc_html_e( 'Shipping', 'custom-checkout' ); ?></button>
      <button id="stepper-3" class="cc-stepper-btn step-inactive" type="button" role="tab" aria-selected="false" aria-controls="step-3"><?php esc_html_e( 'Payment', 'custom-checkout' ); ?></button>
    </nav>

    <form name="checkout" method="post" class="checkout woocommerce-checkout" action="<?php echo esc_url( wc_get_checkout_url() ); ?>" enctype="multipart/form-data">

      <section id="step-1" class="cc-step" role="tabpanel" aria-labelledby="stepper-1">
        <div class="cc-inline-error" role="alert" aria-live="polite" style="display:none"></div>
        <?php cc_render_section_fields( 'billing', $fields['billing'], [
          'limit_country' => $limit_country,
          'limit_state'   => $limit_state,
          'all_countries' => $all_countries,
          'us_states'     => $us_states,
          'counties'      => $counties,
          'county_req'    => $county_req,
          'prefill_email' => $prefill_email,
        ] ); ?>
        <div class="cc-actions">
          <button type="button" id="to-step-2" class="button"><?php esc_html_e( 'Continue to Shipping', 'custom-checkout' ); ?></button>
        </div>
      </section>

      <section id="step-2" class="cc-step is-hidden" role="tabpanel" aria-labelledby="stepper-2">
        <input type="hidden" id="cc_fulfillment_mode" value="pickup" />
        <div class="cc-mode-cards">
          <button type="button" class="cc-mode-card" data-mode="pickup" aria-pressed="true">
            <span class="cc-card-title"><?php esc_html_e( 'Pick up at store', 'custom-checkout' ); ?></span>
            <span class="cc-card-desc"><?php esc_html_e( 'Collect from our location', 'custom-checkout' ); ?></span>
          </button>
          <button type="button" class="cc-mode-card" data-mode="delivery" aria-pressed="false">
            <span class="cc-card-title"><?php esc_html_e( 'Delivery', 'custom-checkout' ); ?></span>
            <span class="cc-card-desc"><?php esc_html_e( 'Ship to your address', 'custom-checkout' ); ?></span>
          </button>
        </div>

        <div id="cc-delivery-note" class="cc-note is-hidden">
          <?php echo wp_kses_post( $delivery_note ); ?>
        </div>

        <div class="cc-inline-error" role="alert" aria-live="polite" style="display:none"></div>

        <div id="cc-shipping-fields" class="is-hidden">
          <?php cc_render_section_fields( 'shipping', $fields['shipping'], [
            'limit_country' => $limit_country,
            'limit_state'   => $limit_state,
            'all_countries' => $all_countries,
            'us_states'     => $us_states,
            'counties'      => $counties,
            'county_req'    => $county_req,
            'prefill_email' => '',
          ] ); ?>
        </div>

        <?php if ( ! empty( $fields['extra'] ) ) : ?>
        <div id="cc-extra-fields">
          <?php cc_render_section_fields( 'extra', $fields['extra'], [
            'limit_country' => '',
            'limit_state'   => '',
            'all_countries' => $all_countries,
            'us_states'     => [],
            'counties'      => [],
            'county_req'    => false,
            'prefill_email' => '',
          ] ); ?>
        </div>
        <?php endif; ?>

        <p><label><?php _e('Order notes (optional)','custom-checkout'); ?><textarea id="cc_order_comments" rows="3" data-section="shipping" data-key="order_comments"></textarea></label></p>

        <div class="cc-actions">
          <button type="button" id="back-to-1" class="button button-secondary"><?php esc_html_e( 'Back', 'custom-checkout' ); ?></button>
          <button type="button" id="to-step-3" class="button"><?php esc_html_e( 'Continue to Payment', 'custom-checkout' ); ?></button>
        </div>
      </section>

      <section id="step-3" class="cc-step is-hidden" role="tabpanel" aria-labelledby="stepper-3">
        <?php
          if ( wc_coupons_enabled() ) { ?>
            <div class="cc-coupon-form woocommerce-form-coupon">
                <p><?php esc_html_e( 'If you have a coupon code, please apply it below.', 'woocommerce' ); ?></p>
                <p class="form-row form-row-first">
                    <input type="text" name="coupon_code" class="input-text" placeholder="<?php esc_attr_e( 'Coupon code', 'woocommerce' ); ?>" id="coupon_code" value="" />
                </p>
                <p class="form-row form-row-last">
                    <button type="button" class="button" name="apply_coupon" value="<?php esc_attr_e( 'Apply coupon', 'woocommerce' ); ?>"><?php esc_html_e( 'Apply coupon', 'woocommerce' ); ?></button>
                </p>
                <div class="clear"></div>
            </div>
          <?php }
          wc_get_template( 'checkout/payment.php', [ 'checkout' => $checkout ] );
        ?>
        <div class="cc-actions">
          <button type="button" id="back-to-2" class="button button-secondary"><?php esc_html_e( 'Back to Shipping', 'custom-checkout' ); ?></button>
        </div>
      </section>

    </form>
  </main>

  <aside class="cc-aside">
    <div class="cc-summary">
      <h3 class="cc-summary-title"><?php esc_html_e( 'Order Summary', 'custom-checkout' ); ?></h3>
      <div id="order_review" class="woocommerce-checkout-review-order">
        <?php wc_get_template( 'checkout/review-order.php', [ 'checkout' => $checkout ] ); ?>
      </div>
    </div>
  </aside>
</div>

<?php do_action( 'woocommerce_after_checkout_form', $checkout ); ?>