/**
 * CPT Admin Updater Script v2.4
 *
 * Handles advanced UI updates, including row coloring, dynamic button states,
 * and API-driven actions for the JetEngine listing grid.
 * This version adds hover tooltips (via PHP), more robust button click detection,
 * and enhanced console logging for easier debugging.
 */
document.addEventListener('DOMContentLoaded', function () {
    console.log('CPT Updater Script v2.4 Loaded.');

    if (typeof cptUpdater === 'undefined') {
        console.error('CRITICAL: cptUpdater object not defined. Check PHP `wp_localize_script`.');
        return;
    }

    // Initialize all listing items on page load
    initializeAllItems();

    // Main event listener for action button clicks
    document.body.addEventListener('click', function (event) {
        // Use .closest() to find the button, even if an inner element (like an icon's span) was clicked.
        const button = event.target.closest('.cpt-action-button');
        if (button) {
            handleButtonClick(button);
        }
    });
});

/**
 * Sets up the initial state for all listing items when the page loads.
 */
function initializeAllItems() {
    const allItems = document.querySelectorAll('.rental-request-item');
    console.log(`Found ${allItems.length} items to initialize. If 0, check that '.rental-request-item' class is on your listing's main container.`);
    allItems.forEach(item => {
        const statusElement = item.querySelector('.rental-status-value');
        if (statusElement) {
            const status = statusElement.textContent.trim().toLowerCase() || 'new';
            updateRowAppearance(item, status);
        } else {
            console.error("Could not find '.rental-status-value' element in item. Make sure this Dynamic Field exists in your listing.", item);
        }
    });
}

/**
 * Handles the logic when an action button is clicked.
 * @param {HTMLElement} button - The button element that was clicked.
 */
function handleButtonClick(button) {
    console.log('--- Button Click Detected ---');
    console.log('Button Element:', button);

    const action = button.dataset.action;
    const postId = button.dataset.postId;
    const listingItem = button.closest('.rental-request-item');

    // --- Start Debugging Logs ---
    console.log(`Action: ${action} (from data-action)`);
    console.log(`Post ID: ${postId} (from data-post-id)`);
    if (listingItem) {
        console.log('Parent .rental-request-item found:', listingItem);
    } else {
        console.error('CRITICAL FAILURE: Could not find parent .rental-request-item. Check your Elementor template structure.');
        return;
    }
    // --- End Debugging Logs ---


    if (!action || !postId) {
        console.error('Button is missing required data-action or data-post-id attributes.');
        return;
    }

    if (button.classList.contains('is-active-status')) {
        console.log('Action aborted: Button already represents the active status.');
        return;
    }

    listingItem.querySelectorAll('.cpt-action-button').forEach(btn => btn.disabled = true);
    
    if (!button.classList.contains('cpt-action-button-icon')) {
        button.textContent = 'Processing...';
    }


    if (action === 'delete') {
        if (confirm('Are you sure you want to permanently delete this item?')) {
            deletePost(button, postId, listingItem);
        } else {
            const statusElement = listingItem.querySelector('.rental-status-value');
            const currentStatus = statusElement ? statusElement.textContent.trim().toLowerCase() : 'new';
            updateRowAppearance(listingItem, currentStatus);
        }
    } else {
        updatePostStatus(button, postId, action, listingItem);
    }
}

/**
 * Central function to update the visual state of a row and its buttons.
 * @param {HTMLElement} item - The main listing item element (.rental-request-item).
 * @param {string} status - The new status ('new', 'pending', 'success', 'cancelled').
 */
function updateRowAppearance(item, status) {
    // Update the hidden status field's text content for future reference
    const statusElement = item.querySelector('.rental-status-value');
    if (statusElement) {
        statusElement.textContent = status;
    }

    // 1. Update Row Background Color
    item.classList.remove('status-new', 'status-pending', 'status-success', 'status-cancelled');
    if (status && status !== 'new') {
        item.classList.add(`status-${status}`);
    }

    // 2. Update Button States
    const buttons = item.querySelectorAll('.cpt-action-button');
    buttons.forEach(btn => {
        const btnAction = btn.dataset.action;
        btn.disabled = false;
        btn.classList.remove('is-active-status', 'status-success', 'status-cancelled', 'status-pending');
        
        // Restore original content (icon or text)
        if (btn.classList.contains('cpt-action-button-icon')) {
            // It's an icon button, do nothing to the content
        } else if (btn.dataset.label) {
            // It's a text button, restore its label
            btn.textContent = btn.dataset.label;
        }

        if (btnAction === status) {
            btn.classList.add('is-active-status', `status-${status}`);
            
            // If it's a text button, update its text to show the status.
            if (!btn.classList.contains('cpt-action-button-icon')) {
                 btn.textContent = status.charAt(0).toUpperCase() + status.slice(1);
            }

            btn.disabled = true;
        }
    });
}

/**
 * Updates the 'request_status' meta field via REST API.
 * @param {HTMLElement} button - The button that was clicked.
 * @param {string} postId - The ID of the post to update.
 * @param {string} newStatus - The new status to set.
 * @param {HTMLElement} listingItem - The parent listing item.
 */
function updatePostStatus(button, postId, newStatus, listingItem) {
    const apiEndpoint = `${cptUpdater.rest_url}wp/v2/${cptUpdater.cpt_slug}/${postId}`;

    fetch(apiEndpoint, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json', 'X-WP-Nonce': cptUpdater.nonce },
        body: JSON.stringify({ meta: { request_status: newStatus } }),
    })
    .then(response => {
        if (!response.ok) {
            return response.json().then(err => { throw new Error(err.message || 'API Error'); });
        }
        return response.json();
    })
    .then(data => {
        console.log('Update Success:', data);
        updateRowAppearance(listingItem, newStatus);
    })
    .catch(error => {
        console.error('Error updating post:', error.message);
        alert('Error: ' + error.message);
        const statusElement = listingItem.querySelector('.rental-status-value');
        const currentStatus = statusElement ? statusElement.textContent.trim().toLowerCase() : 'new';
        updateRowAppearance(listingItem, currentStatus);
    });
}

/**
 * Deletes a post via REST API.
 * @param {HTMLElement} button - The button that was clicked.
 * @param {string} postId - The ID of the post to delete.
 * @param {HTMLElement} listingItem - The parent listing item.
 */
function deletePost(button, postId, listingItem) {
    const apiEndpoint = `${cptUpdater.rest_url}wp/v2/${cptUpdater.cpt_slug}/${postId}?force=true`;

    fetch(apiEndpoint, {
        method: 'DELETE',
        headers: { 'X-WP-Nonce': cptUpdater.nonce },
    })
    .then(response => {
        if (!response.ok) {
            return response.json().then(err => { throw new Error(err.message || 'API Error'); });
        }
        return response;
    })
    .then(() => {
        listingItem.style.transition = 'opacity 0.5s ease';
        listingItem.style.opacity = '0';
        setTimeout(() => listingItem.remove(), 500);
    })
    .catch(error => {
        console.error('Error deleting post:', error.message);
        alert('Error: ' + error.message);
        const statusElement = listingItem.querySelector('.rental-status-value');
        const currentStatus = statusElement ? statusElement.textContent.trim().toLowerCase() : 'new';
        updateRowAppearance(listingItem, currentStatus);
    });
}
