<?php
/**
 * Plugin Name:       Custom WooCommerce Checkout
 * Description:       Custom 3-step checkout UI with our own fields via REST; native Woo payment step.
 * Version:           1.1.0
 * Author:            Your Name
 * License:           GPL-2.0+
 * Text Domain:       custom-checkout
 */

if ( ! defined( 'ABSPATH' ) ) exit;

final class Custom_Checkout_Plugin {
    private static $instance = null;
    const SLUG = 'custom-checkout';
    const VER  = '1.1.0';

    public static function instance() {
        return self::$instance ?? ( self::$instance = new self() );
    }

    private function __construct() {
        add_action( 'plugins_loaded', [ $this, 'maybe_init' ] );
    }

    public function maybe_init() {
        if ( ! class_exists( 'WooCommerce' ) ) {
            add_action( 'admin_notices', function () {
                echo '<div class="notice notice-error"><p><strong>Custom Checkout</strong> requires WooCommerce.</p></div>';
            } );
            return;
        }

        add_action( 'wp_enqueue_scripts', [ $this, 'enqueue_assets' ], 20 );
        add_filter( 'woocommerce_locate_template', [ $this, 'load_templates' ], 10, 3 );

        // Our REST API endpoints
        add_action( 'rest_api_init', [ $this, 'register_routes' ] );

        // Inject saved data into Woo’s posted checkout data right before processing payment
        add_filter( 'woocommerce_checkout_posted_data', [ $this, 'merge_saved_data_into_checkout' ] );

        // Persist custom meta on the order (e.g., fulfillment mode)
        add_action( 'woocommerce_checkout_create_order', [ $this, 'save_custom_meta' ], 10, 2 );
    }

    public function enqueue_assets() {
        if ( ! is_checkout() || is_order_received_page() ) return;

        $url = plugin_dir_url( __FILE__ );
        wp_enqueue_style( self::SLUG . '-css', $url . 'assets/css/checkout-style.css', [], self::VER );
        wp_enqueue_script( self::SLUG . '-js',  $url . 'assets/js/checkout-script.js', [ 'jquery','wc-checkout' ], self::VER, true );
        wp_localize_script( self::SLUG . '-js', 'CustomCheckout', [
            'nonce'   => wp_create_nonce( 'wp_rest' ),
            'restUrl' => esc_url_raw( rest_url( 'custom-checkout/v1' ) ),
        ] );
    }

    public function load_templates( $template, $template_name, $template_path ) {
        $base = plugin_dir_path( __FILE__ ) . 'templates/woocommerce/';
        $candidate = $base . $template_name;
        return file_exists( $candidate ) ? $candidate : $template;
    }

    /* ---------------- REST: save data from our UI ---------------- */

    public function register_routes() {
        register_rest_route( 'custom-checkout/v1', '/save', [
            'methods'  => 'POST',
            'callback' => [ $this, 'rest_save' ],
            'permission_callback' => function () { return true; }, // guest checkout allowed; rely on nonce
            'args' => [
                'section' => [ 'required' => true, 'type' => 'string', 'enum' => [ 'billing', 'shipping', 'mode' ] ],
                'data'    => [ 'required' => true, 'type' => 'object' ],
            ],
        ] );
    }

    public function rest_save( WP_REST_Request $req ) {
        if ( ! wp_verify_nonce( $req->get_header( 'x-wp-nonce' ), 'wp_rest' ) ) {
            return new WP_REST_Response( [ 'ok' => false, 'message' => 'Bad nonce' ], 403 );
        }

        if ( ! function_exists( 'WC' ) || ! WC()->session ) {
            return new WP_REST_Response( [ 'ok' => false, 'message' => 'No Woo session' ], 500 );
        }

        $section = $req->get_param( 'section' );
        $data    = (array) $req->get_param( 'data' );

        // Sanitize the subset we expect (expand as you add fields)
        $clean = [];

        switch ( $section ) {
            case 'billing':
                $map = [
                    'first_name','last_name','email','phone',
                    'company','address_1','address_2','city','state','postcode','country',
                ];
                foreach ( $map as $k ) {
                    $val = isset($data[$k]) ? wc_clean( wp_unslash( $data[$k] ) ) : '';
                    $clean["billing_{$k}"] = $val;
                }
                break;

            case 'shipping':
                $map = [ 'first_name','last_name','company','address_1','address_2','city','state','postcode','country' ];
                foreach ( $map as $k ) {
                    $val = isset($data[$k]) ? wc_clean( wp_unslash( $data[$k] ) ) : '';
                    $clean["shipping_{$k}"] = $val;
                }
                // notes are optional
                if ( isset( $data['order_comments'] ) ) {
                    $clean['order_comments'] = wc_clean( wp_unslash( $data['order_comments'] ) );
                }
                break;

            case 'mode':
                $mode = isset( $data['fulfillment_mode'] ) && in_array( $data['fulfillment_mode'], [ 'pickup','delivery' ], true )
                    ? $data['fulfillment_mode'] : 'pickup';
                $clean['cc_fulfillment_mode'] = $mode;
                break;
        }

        // Save in Woo session (source of truth for our flow)
        $saved = (array) WC()->session->get( 'cc_checkout', [] );
        $saved = array_merge( $saved, $clean );
        WC()->session->set( 'cc_checkout', $saved );

        // Also update WC Customer so shipping/tax/rates reflect immediately
        $customer = WC()->customer;
        if ( $customer && is_a( $customer, 'WC_Customer' ) ) {
            foreach ( $saved as $key => $val ) {
                if ( strpos( $key, 'billing_' ) === 0 ) { $customer->{"set_{$key}"}( $val ); }
                if ( strpos( $key, 'shipping_' ) === 0 ) { $customer->{"set_{$key}"}( $val ); }
            }
            $customer->save();
        }

        // Force totals regeneration on the client
        WC()->cart->calculate_totals();

        return new WP_REST_Response( [ 'ok' => true, 'saved' => $saved ], 200 );
    }

    /* --------- Merge saved fields into Woo’s checkout data ---------- */

    public function merge_saved_data_into_checkout( $posted ) {
        if ( ! WC()->session ) return $posted;
        $saved = (array) WC()->session->get( 'cc_checkout', [] );
        if ( ! $saved ) return $posted;

        // Ensure payment step has all standard fields even if our UI hid them
        foreach ( $saved as $key => $val ) {
            if ( $val === '' || $val === null ) continue;
            $posted[ $key ] = $val;
        }

        // If mode is pickup, blank out shipping to avoid unwanted taxes (optional)
        if ( isset( $saved['cc_fulfillment_mode'] ) && $saved['cc_fulfillment_mode'] === 'pickup' ) {
            $posted['ship_to_different_address'] = 0;
        } else {
            $posted['ship_to_different_address'] = 1;
        }

        return $posted;
    }

    public function save_custom_meta( $order, $data ) {
        if ( ! WC()->session ) return;
        $saved = (array) WC()->session->get( 'cc_checkout', [] );
        if ( isset( $saved['cc_fulfillment_mode'] ) ) {
            $order->update_meta_data( '_cc_fulfillment_mode', $saved['cc_fulfillment_mode'] );
        }
    }
}
Custom_Checkout_Plugin::instance();
